<?php

class DeeLauLab_LautFM_Admin {

    private $option_name = 'deelaulab_lfh_settings';

    public function __construct() {
        add_action('admin_menu', [$this, 'add_plugin_page']);
        add_action('admin_init', [$this, 'page_init']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }

    public function enqueue_admin_assets($hook) {
        if ($hook != 'toplevel_page_deelaulab-lautfm-history') {
            return;
        }

        wp_enqueue_style('wp-color-picker');
        wp_enqueue_style('deelaulab-lautfm-admin-style', plugin_dir_url(dirname(__FILE__)) . 'assets/css/admin-style.css', [], '1.0.0');
        wp_enqueue_script('wp-color-picker-alpha', plugin_dir_url(dirname(__FILE__)) . 'assets/js/wp-color-picker-alpha.js', ['jquery', 'wp-color-picker'], '3.0.4', true);
        wp_add_inline_script('wp-color-picker-alpha', 'jQuery(document).ready(function($){$(".wp-color-picker-field").wpColorPicker();});');
    }

    public function add_plugin_page() {
        add_menu_page(
            'Laut.fm History Einstellungen',
            'DLL-History',
            'manage_options',
            'deelaulab-lautfm-history',
            [$this, 'create_admin_page'],
            'dashicons-format-audio',
            58
        );
    }

    public function create_admin_page() {
        if (isset($_GET['settings-updated'])) {
            add_settings_error('dllfh_messages', 'dllfh_message', __('Einstellungen gespeichert.', 'deelaulab-lautfm-history'), 'updated');
        }
        if (isset($_GET['cache-cleared'])) {
            $count = isset($_GET['count']) ? absint($_GET['count']) : 0;
            add_settings_error('dllfh_messages', 'dllfh_message', sprintf(__('Cover-Cache erfolgreich geleert. %d Einträge wurden entfernt.', 'deelaulab-lautfm-history'), $count), 'updated');
        }

        settings_errors('dllfh_messages');
        ?>
        <div class="wrap dllfh-settings-wrapper">
            <h1>Laut.fm History Einstellungen</h1>
            
            <form method="post" action="">
                <?php
                wp_nonce_field('dllfh_save_settings', 'dllfh_save_settings_nonce');
                do_settings_sections('deelaulab-lfh-admin');
                submit_button();
                ?>
            </form>

            <hr>

            <h2>Cache-Verwaltung</h2>
            <p>Wenn ein Cover nicht angezeigt wird, obwohl es existieren sollte, leeren Sie hier den Cache. Das Plugin wird dann gezwungen, für alle Songs erneut online zu suchen.</p>
            <form method="post" action="">
                <?php wp_nonce_field('dllfh_clear_cache', 'dllfh_clear_cache_nonce'); ?>
                <input type="hidden" name="action" value="dllfh_clear_cache">
                <?php submit_button('Cover-Cache jetzt leeren', 'delete', 'submit_clear_cache'); ?>
            </form>
        </div>
        <?php
    }

    public function handle_form_save() {
        if (isset($_POST['dllfh_save_settings_nonce']) && wp_verify_nonce($_POST['dllfh_save_settings_nonce'], 'dllfh_save_settings')) {
            if (!current_user_can('manage_options')) return;
            
            $raw_options = $_POST[$this->option_name] ?? [];
            $sanitized_options = $this->sanitize($raw_options);
            update_option($this->option_name, $sanitized_options);
            
            wp_redirect(admin_url('admin.php?page=deelaulab-lautfm-history&settings-updated=true'));
            exit;
        }
        
        if (isset($_POST['dllfh_clear_cache_nonce']) && wp_verify_nonce($_POST['dllfh_clear_cache_nonce'], 'dllfh_clear_cache')) {
            if (!current_user_can('manage_options')) return;
            
            global $wpdb;
            $prefix = '_transient_dllfh_cover_';
            $sql = "SELECT `option_name` FROM `{$wpdb->options}` WHERE `option_name` LIKE '%s'";
            $transients = $wpdb->get_col($wpdb->prepare($sql, $wpdb->esc_like($prefix) . '%'));
            
            $count = 0;
            foreach ($transients as $transient) {
                $transient_name = str_replace('_transient_', '', $transient);
                if (delete_transient($transient_name)) {
                    $count++;
                }
            }
            
            wp_redirect(admin_url('admin.php?page=deelaulab-lautfm-history&cache-cleared=true&count=' . $count));
            exit;
        }
    }

    public function page_init() {
        $this->handle_form_save();
        
        add_settings_section('setting_section_api', 'API-Schlüssel (Optional)', [$this, 'print_api_section_info'], 'deelaulab-lfh-admin');
        add_settings_section('setting_section_colors', 'Farb-Einstellungen', null, 'deelaulab-lfh-admin');
        add_settings_section('setting_section_layout', 'Layout & Inhalt', null, 'deelaulab-lfh-admin');
        add_settings_section('setting_section_debug', 'Fehlersuche (Debug)', null, 'deelaulab-lfh-admin');

        $fields = [
            'spotify_client_id'   => ['section' => 'setting_section_api', 'title' => 'Spotify Client ID', 'type' => 'text'],
            'spotify_client_secret' => ['section' => 'setting_section_api', 'title' => 'Spotify Client Secret', 'type' => 'password'],
            'lastfm_api_key'      => ['section' => 'setting_section_api', 'title' => 'Last.fm API Key', 'type' => 'text'],
            'fallback_cover_url'  => ['section' => 'setting_section_api', 'title' => 'Fallback Cover URL', 'type' => 'text', 'desc' => 'URL zu einem Standard-Cover, wenn keins gefunden wird.'],
            'bg_color_main'       => ['section' => 'setting_section_colors', 'title' => 'Gesamt Hintergrund', 'type' => 'color'],
            'bg_color_song'       => ['section' => 'setting_section_colors', 'title' => 'Song Hintergrund', 'type' => 'color'],
            'color_title'         => ['section' => 'setting_section_colors', 'title' => 'Song-Titel Schriftfarbe', 'type' => 'color'],
            'color_artist'        => ['section' => 'setting_section_colors', 'title' => 'Interpret Schriftfarbe', 'type' => 'color'],
            'color_time'          => ['section' => 'setting_section_colors', 'title' => 'Zeit Schriftfarbe', 'type' => 'color'],
            'cover_size'          => ['section' => 'setting_section_layout', 'title' => 'Cover-Größe (in px)', 'type' => 'number', 'desc' => 'Empfehlung: 80'],
            'history_count'       => ['section' => 'setting_section_layout', 'title' => 'Anzahl Einträge (1-10)', 'type' => 'number_range', 'min' => 1, 'max' => 10],
            'show_icons'          => ['section' => 'setting_section_layout', 'title' => 'Angezeigte Icons', 'type' => 'icons'],
            'debug_mode'          => ['section' => 'setting_section_debug', 'title' => 'Debug-Modus aktivieren', 'type' => 'checkbox', 'desc' => 'Zeigt die geladenen Einstellungen direkt über dem Shortcode an.']
        ];

        foreach ($fields as $id => $field) {
            add_settings_field($id, $field['title'], [$this, 'render_field'], 'deelaulab-lfh-admin', $field['section'], ['id' => $id, 'field' => $field]);
        }
    }

    public function print_api_section_info() {
        echo '<p>Tragen Sie hier Ihre eigenen API-Schlüssel ein, um die Cover-Suche zu verbessern. Wenn die Felder leer bleiben, werden öffentliche APIs (Deezer, iTunes) bevorzugt.</p>';
    }
    public function sanitize($input) {
        if (!is_array($input)) {
            return [];
        }

        $sanitized_output = [];
        $defaults = (new DeeLauLab_LautFM_Controller())->get_default_settings();
        $merged_input = wp_parse_args($input, $defaults);

        $sanitized_output['spotify_client_id'] = sanitize_text_field($merged_input['spotify_client_id']);
        $sanitized_output['spotify_client_secret'] = sanitize_text_field($merged_input['spotify_client_secret']);
        $sanitized_output['lastfm_api_key'] = sanitize_text_field($merged_input['lastfm_api_key']);
        $sanitized_output['fallback_cover_url'] = esc_url_raw($merged_input['fallback_cover_url']);
        $sanitized_output['bg_color_main'] = $this->sanitize_rgba_or_hex_color($merged_input['bg_color_main']);
        $sanitized_output['bg_color_song'] = $this->sanitize_rgba_or_hex_color($merged_input['bg_color_song']);
        $sanitized_output['color_title'] = $this->sanitize_rgba_or_hex_color($merged_input['color_title']);
        $sanitized_output['color_artist'] = $this->sanitize_rgba_or_hex_color($merged_input['color_artist']);
        $sanitized_output['color_time'] = $this->sanitize_rgba_or_hex_color($merged_input['color_time']);
        $sanitized_output['cover_size'] = absint($merged_input['cover_size']);
        $sanitized_output['history_count'] = absint($merged_input['history_count']);

        if ($sanitized_output['history_count'] > 10) $sanitized_output['history_count'] = 10;
        if ($sanitized_output['history_count'] < 1) $sanitized_output['history_count'] = 1;

        $sanitized_output['show_icons'] = isset($input['show_icons']) && is_array($input['show_icons']) ? array_map('sanitize_key', $input['show_icons']) : [];
        $sanitized_output['debug_mode'] = isset($input['debug_mode']) && $input['debug_mode'] === '1' ? '1' : '0';
        
        return $sanitized_output;
    }
    private function sanitize_rgba_or_hex_color($color) {
        if (empty($color)) {
            return '';
        }
        $color = trim($color);
        if (strpos($color, 'rgba') === 0) {
            if (preg_match('/^rgba\((\d{1,3}),\s*(\d{1,3}),\s*(\d{1,3}),\s*([0-9\.]+)\)$/', $color)) {
                return $color;
            }
        }
        if (strpos($color, 'rgb') === 0) {
            if (preg_match('/^rgb\((\d{1,3}),\s*(\d{1,3}),\s*(\d{1,3})\)$/', $color)) {
                return $color;
            }
        }
        return sanitize_hex_color($color);
    }
    public function render_field($args) {
        $id = $args['id'];
        $field = $args['field'];
        $defaults = (new DeeLauLab_LautFM_Controller())->get_default_settings();
        $options = get_option($this->option_name, $defaults);
        $value = $options[$id] ?? $defaults[$id];

        switch ($field['type']) {
            case 'text':
                printf('<input type="text" id="%s" name="%s[%s]" value="%s" class="regular-text" />', $id, $this->option_name, $id, esc_attr($value));
                break;

            case 'password':
                printf('<input type="password" id="%s" name="%s[%s]" value="%s" class="regular-text" />', $id, $this->option_name, $id, esc_attr($value));
                break;

            case 'color':
                printf('<input type="text" id="%s" name="%s[%s]" value="%s" class="wp-color-picker-field" data-alpha-enabled="true" />', $id, $this->option_name, $id, esc_attr($value));
                break;

            case 'number':
                printf('<input type="number" id="%s" name="%s[%s]" value="%s" />', $id, $this->option_name, $id, esc_attr($value));
                break;

            case 'number_range':
                printf('<input type="number" id="%s" name="%s[%s]" value="%s" min="%d" max="%d" />', $id, $this->option_name, $id, esc_attr($value), $field['min'], $field['max']);
                break;

            case 'checkbox':
                printf('<input type="checkbox" id="%s" name="%s[%s]" value="1" %s />', $id, $this->option_name, $id, checked('1', $value, false));
                break;
                
            case 'icons':
                $services = ['youtube' => 'YouTube', 'soundcloud' => 'SoundCloud', 'spotify' => 'Spotify', 'amazon' => 'Amazon Music', 'itunes' => 'Apple Music'];
                echo '<div class="icon-toggle-list">';
                foreach ($services as $key => $label) {
                    $checked = (is_array($value) && isset($value[$key]) && $value[$key] === '1') ? 'checked' : '';
                    printf('<label><input type="checkbox" name="%s[show_icons][%s]" value="1" %s /> %s</label>', $this->option_name, $key, $checked, $label);
                }
                echo '</div>';
                break;
        }

        if (!empty($field['desc'])) {
            printf('<p class="description">%s</p>', $field['desc']);
        }
    }
}