<?php
/**
 * Klasse für alle Frontend-Funktionen (Shortcodes, Formularverarbeitung).
 */
class DLLW_Frontend {

    private static $instance = null;

    public static function get_instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'init', [ $this, 'register_shortcode' ] );
        add_action( 'init', [ $this, 'handle_form_submission' ] );
        add_action( 'init', [ $this, 'handle_widget_request' ] );
        add_action( 'template_redirect', [ $this, 'handle_frontend_wish_actions' ] );
    }

    public function register_shortcode() {
        add_shortcode( 'deelaulab_wunschbox_formular', [ $this, 'render_form_shortcode' ] );
    }

    public function render_form_shortcode() {
        ob_start();

        if ( isset( $_GET['dllw_success'] ) && $_GET['dllw_success'] === '1' ) {
            echo '<p style="color: green;"><strong>' . __( 'Vielen Dank, dein Wunsch wurde erfolgreich übermittelt!', 'deelaulab-wunschbox' ) . '</strong></p>';
        }
        if ( isset( $_GET['dllw_error'] ) ) {
            echo '<p style="color: red;"><strong>' . esc_html( urldecode( $_GET['dllw_error'] ) ) . '</strong></p>';
        }

        include_once DLLW_PLUGIN_PATH . 'views/frontend-form.php';

        return ob_get_clean();
    }

    public function handle_form_submission() {
        if ( ! isset( $_POST['dllw_form_submit'] ) ) {
            return;
        }

        if ( ! isset( $_POST['dllw_nonce'] ) || ! wp_verify_nonce( $_POST['dllw_nonce'], 'dllw_add_wunsch_action' ) ) {
            wp_die( 'Sicherheitsüberprüfung fehlgeschlagen.' );
        }

        // Daten aus dem Formular holen und bereinigen
        $sender_name = isset( $_POST['dllw_sender_name'] ) ? sanitize_text_field( $_POST['dllw_sender_name'] ) : '';
        $artist      = isset( $_POST['dllw_artist'] ) ? sanitize_text_field( $_POST['dllw_artist'] ) : '';
        $song_title  = isset( $_POST['dllw_song_title'] ) ? sanitize_text_field( $_POST['dllw_song_title'] ) : '';
        $message     = isset( $_POST['dllw_message'] ) ? sanitize_textarea_field( $_POST['dllw_message'] ) : '';

        if ( empty( $sender_name ) || empty( $message ) ) {
            $error_message = urlencode( 'Bitte gib deinen Namen und einen Grußtext an.' );
            wp_safe_redirect( add_query_arg( 'dllw_error', $error_message, wp_get_referer() ) );
            exit;
        }

        // Kombinierten Titel für die Hauptanzeige erstellen
        $combined_title = 'Gruß'; // Standard, falls kein Song gewünscht
        if ( ! empty( $artist ) && ! empty( $song_title ) ) {
            $combined_title = $artist . ' - ' . $song_title;
        } elseif ( ! empty( $song_title ) ) {
            $combined_title = $song_title;
        } elseif ( ! empty( $artist ) ) {
            $combined_title = $artist;
        }

        $post_data = [
            'post_type'    => 'dllw_wunsch',
            'post_title'   => $combined_title,
            'post_content' => $message,
            'post_status'  => 'wartend',
        ];

        $new_post_id = wp_insert_post( $post_data, true );

        if ( is_wp_error( $new_post_id ) ) {
            $error_message = urlencode( 'Fehler beim Speichern des Wunsches.' );
            wp_safe_redirect( add_query_arg( 'dllw_error', $error_message, wp_get_referer() ) );
            exit;
        }

        // Speichere die einzelnen Felder als Metadaten
        update_post_meta( $new_post_id, '_dllw_sender_name', $sender_name );
        update_post_meta( $new_post_id, '_dllw_artist', $artist );
        update_post_meta( $new_post_id, '_dllw_song_title', $song_title );

        wp_safe_redirect( add_query_arg( 'dllw_success', '1', wp_get_referer() ) );
        exit;
    }
    
    public function handle_widget_request() {
        if ( ! isset( $_GET['deelaulab_widget'] ) || $_GET['deelaulab_widget'] !== 'true' ) {
            return;
        }

        $submitted_token = isset( $_GET['token'] ) ? sanitize_text_field( $_GET['token'] ) : '';
        $stored_token = get_option( 'dllw_widget_token' );

        if ( empty( $stored_token ) || ! hash_equals( $stored_token, $submitted_token ) ) {
            wp_die( 'Ungültiger oder fehlender Sicherheitstoken.' );
        }

        if ( isset( $_GET['action'] ) && isset( $_GET['wunsch_id'] ) ) {
            $nonce = isset( $_GET['_wpnonce'] ) ? $_GET['_wpnonce'] : null;
            $wunsch_id = intval( $_GET['wunsch_id'] );

            if ( wp_verify_nonce( $nonce, 'dllw_widget_action_' . $wunsch_id ) ) {
                $action = sanitize_key( $_GET['action'] );
                switch ( $action ) {
                    case 'set_status_gespielt':
                        wp_update_post( [ 'ID' => $wunsch_id, 'post_status' => 'gespielt' ] );
                        break;
                    case 'trash':
                        wp_trash_post( $wunsch_id );
                        break;
                }
            }
            wp_safe_redirect( home_url( '/?deelaulab_widget=true&token=' . $stored_token ) );
            exit;
        }
        
        $wartende_wuensche = get_posts([
            'post_type'   => 'dllw_wunsch',
            'post_status' => ['wartend', 'wird_gesucht'],
            'numberposts' => -1,
            'orderby'     => 'date',
            'order'       => 'ASC',
        ]);

        include_once DLLW_PLUGIN_PATH . 'views/external-widget.php';
        
        exit;
    }
    public function handle_frontend_wish_actions() {
        if ( ! isset( $_GET['dllw_action'] ) || ! isset( $_GET['wunsch_id'] ) ) {
            return;
        }
        $nonce = isset( $_GET['_wpnonce'] ) ? $_GET['_wpnonce'] : '';
        if ( ! wp_verify_nonce( $nonce, 'dllw_frontend_action_' . intval( $_GET['wunsch_id'] ) ) ) {
            wp_die( 'Ungültige Anfrage.' );
        }
        if ( ! current_user_can( 'manage_wishes' ) ) {
            wp_die( 'Du hast keine Berechtigung für diese Aktion.' );
        }

        $action    = sanitize_key( $_GET['dllw_action'] );
        $wunsch_id = intval( $_GET['wunsch_id'] );
        switch ( $action ) {
            case 'set_status_gesucht':
                wp_update_post( [ 'ID' => $wunsch_id, 'post_status' => 'wird_gesucht' ] );
                break;
            case 'set_status_gespielt':
                wp_update_post( [ 'ID' => $wunsch_id, 'post_status' => 'gespielt' ] );
                break;
            case 'trash':
                wp_trash_post( $wunsch_id );
                break;
        }
        wp_safe_redirect( remove_query_arg( [ 'dllw_action', 'wunsch_id', '_wpnonce' ] ) );
        exit;
    }
    /**
     * Registriert den Shortcode [deelaulab_wunschbox_moderation].
     */
    public function register_moderator_shortcode() {
        add_shortcode( 'deelaulab_wunschbox_moderation', [ $this, 'render_moderator_view_shortcode' ] );
    }

    /**
     * Stellt die HTML-Ansicht für den Moderator-Shortcode bereit.
     * @return string Der HTML-Code der Wunschliste oder eine Fehlermeldung.
     */
    public function render_moderator_view_shortcode() {
        // Nur Benutzer mit der richtigen Berechtigung dürfen dies sehen.
        if ( ! is_user_logged_in() || ! current_user_can( 'manage_wishes' ) ) {
            return '<p>Du hast keine Berechtigung, diesen Bereich zu sehen.</p>';
        }

        // Lade alle aktiven Wünsche
        $wartende_wuensche = get_posts([
            'post_type'   => 'dllw_wunsch',
            'post_status' => ['wartend', 'wird_gesucht'],
            'numberposts' => -1,
            'orderby'     => 'date',
            'order'       => 'ASC',
        ]);

        ob_start();
        ?>
        <div class="dllw-moderator-view">
            <h3>Aktive Wünsche</h3>
            <?php if ( ! empty( $wartende_wuensche ) ) : ?>
                <table style="width: 100%; border-collapse: collapse;">
                    <thead>
                        <tr style="text-align: left;">
                            <th style="padding: 8px; border-bottom: 1px solid #ddd;">Von</th>
                            <th style="padding: 8px; border-bottom: 1px solid #ddd;">Wunsch / Gruss</th>
                            <th style="padding: 8px; border-bottom: 1px solid #ddd;">Aktionen</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $wartende_wuensche as $wunsch ) : ?>
                            <?php
                                $nonce = wp_create_nonce( 'dllw_frontend_action_' . $wunsch->ID );
                                $url_gesucht = add_query_arg( ['dllw_action' => 'set_status_gesucht', 'wunsch_id' => $wunsch->ID, '_wpnonce' => $nonce] );
                                $url_gespielt = add_query_arg( ['dllw_action' => 'set_status_gespielt', 'wunsch_id' => $wunsch->ID, '_wpnonce' => $nonce] );
                                $url_trash = add_query_arg( ['dllw_action' => 'trash', 'wunsch_id' => $wunsch->ID, '_wpnonce' => $nonce] );
                            ?>
                            <tr style="border-bottom: 1px solid #eee;">
                                <td style="padding: 8px;"><strong><?php echo esc_html( get_post_meta( $wunsch->ID, '_dllw_sender_name', true ) ); ?></strong></td>
                                <td style="padding: 8px;">
                                    <strong><?php echo esc_html( $wunsch->post_title ); ?></strong>
                                    <p style="margin: 5px 0 0;"><?php echo nl2br( esc_html( $wunsch->post_content ) ); ?></p>
                                </td>
                                <td style="padding: 8px;">
                                    <?php if ( $wunsch->post_status === 'wartend' ) : ?>
                                        <a href="<?php echo esc_url( $url_gesucht ); ?>">Suchen</a> |
                                    <?php endif; ?>
                                    <a href="<?php echo esc_url( $url_gespielt ); ?>">Gespielt</a> |
                                    <a href="<?php echo esc_url( $url_trash ); ?>" style="color: #d63638;">Löschen</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else : ?>
                <p>Aktuell liegen keine Wünsche vor.</p>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }
}